from datetime import datetime, timedelta, timezone
from flask import Flask, request, render_template, redirect, url_for, make_response
from flask_sqlalchemy import SQLAlchemy

app = Flask(__name__)
app.config['SQLALCHEMY_DATABASE_URI'] = 'sqlite:///users.db'
db = SQLAlchemy(app)

# User model
class User(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    username = db.Column(db.String, unique=True, nullable=False)
    password = db.Column(db.String, nullable=False)
    name = db.Column(db.String, nullable=False)
    email = db.Column(db.String, unique=True, nullable=False)

# Session Model
class Session(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    user_id = db.Column(db.Integer, db.ForeignKey(User.id), nullable=False)
    active = db.Column(db.Boolean, default=True, nullable=False)
    expires_at = db.Column(db.String, nullable=False)

# Home/Login Page
@app.route("/", methods=["GET"])
def login_get():
    return render_template("login.html", error=None)

# Home/Login Page
@app.route("/", methods=["POST"])
def login_post():
    # Get login credentials
    username = request.form["username"]
    password = request.form["password"]

    # Get user from DB
    user = User.query.filter(User.username==username, User.password==password).first()

    # If no user found, show error on login page
    if not user:
        return render_template("login.html", error="Invalid username or password")

    # Create new session
    expires_at = (datetime.now(timezone.utc) + timedelta(hours=1)).isoformat()   # Set expiry time (1 hour)
    new_session = Session(user_id=user.id, expires_at=expires_at)
    db.session.add(new_session)
    db.session.commit()

    # Set session cookie and redirect to dashboard
    response = make_response(redirect(url_for("dashboard")))
    response.set_cookie("session_id", str(new_session.id))
    return response

# Protected Dashboard Route
@app.route("/dashboard")
def dashboard():
    # Check that request contains a session id cookie
    session_id = request.cookies.get("session_id")
    if not session_id:
        return redirect(url_for("login_get"))

    # Check that corresponding session exists in DB
    session = Session.query.filter(Session.id==int(session_id)).first()
    if not session:
        return redirect(url_for("login_get"))
        
    # Check that session is valid
    if datetime.now(timezone.utc) >  datetime.fromisoformat(session.expires_at):  # Check expiration
        return redirect(url_for("login_get"))

    # Valid session so get the corresponding user details
    user = User.query.filter(User.id==session.user_id).first()

    return render_template("dashboard.html", user=user)

# Logout Route
@app.route("/logout")
def logout():
    session_id = request.cookies.get("session_id")

    # Set session to inactive in database
    if not session_id:
        session = Session.query.filter(Session.id==int(session_id)).first()
        session.active = False
        db.session.commit()

    # Instruct browser to destruct cookie and redirect to login page
    response = make_response(redirect(url_for("login_get")))
    response.set_cookie("session_id", "", expires=0)  # Delete session cookie
    return response

app.run(debug=True, reloader_type='stat', port=5000)
