from typing import List
from cryptography.hazmat.primitives.asymmetric import rsa, padding as asym_padding
from cryptography.hazmat.primitives.asymmetric.utils import Prehashed
from cryptography.hazmat.primitives import hashes, serialization
import base64
from cryptography.fernet import Fernet

class Symmetric:
    @staticmethod
    def generate_key() -> str:
        # Returns a Base64-encoded key as an ASCII string
        return Fernet.generate_key().decode('ascii')

    @staticmethod
    def encrypt(message: str, key: str) -> str:
        if not isinstance(message, str):
            raise TypeError("message must be a string")
        if not isinstance(key, str):
            raise TypeError("key must be a string")
        if len(message) == 0:
            raise ValueError("message must not be empty")

        try:
            message_bytes = message.encode('ascii')
        except Exception as e:
            raise ValueError("Failed to encode message in ASCII") from e

        try:
            key_bytes = key.encode('ascii')
        except Exception as e:
            raise ValueError("Failed to encode key in ASCII") from e

        return Fernet(key_bytes).encrypt(message_bytes).decode('ascii')

    @staticmethod
    def decrypt(encrypted: str, key: str) -> str:
        if not isinstance(encrypted, str):
            raise TypeError("encrypted must be a string")
        if not isinstance(key, str):
            raise TypeError("key must be a string")
        
        try:
            encrypted_bytes = encrypted.encode("ascii")
        except Exception as e:
            raise ValueError("Failed to encode encrypted message in ASCII") from e

        try:
            key_bytes = key.encode('ascii')
        except Exception as e:
            raise ValueError("Failed to encode key in ASCII") from e

        return Fernet(key_bytes).decrypt(encrypted_bytes).decode('ascii')


class Asymmetric:
    @staticmethod
    def generate_keys() -> List[str]:
        # Generate a new RSA key pair
        private_key_obj = rsa.generate_private_key(
            public_exponent=65537,
            key_size=1024
        )
        public_key_obj = private_key_obj.public_key()

        # Serialize the private key to DER format
        der_private = private_key_obj.private_bytes(
            encoding=serialization.Encoding.DER,
            format=serialization.PrivateFormat.PKCS8,
            encryption_algorithm=serialization.NoEncryption()
        )

        # Serialize the public key to DER format
        der_public = public_key_obj.public_bytes(
            encoding=serialization.Encoding.DER,
            format=serialization.PublicFormat.SubjectPublicKeyInfo
        )

        # Encode the DER bytes into Base64 strings
        b64_private = base64.b64encode(der_private).decode('utf-8')
        b64_public = base64.b64encode(der_public).decode('utf-8')

        return b64_private, b64_public

    @staticmethod
    def encrypt(message: str, public_key: str) -> str:
        if not isinstance(message, str):
            raise TypeError("message must be a string")
        if not isinstance(public_key, str):
            raise TypeError("public_key must be a string")

        message_bytes = message.encode('utf-8')


        # Validate and decode the Base64 public key string
        try:
            der_public_bytes = base64.b64decode(public_key)
        except Exception as e:
            raise ValueError("public_key is not valid Base64") from e

        try:
            loaded_public_key = serialization.load_der_public_key(der_public_bytes)
        except Exception as e:
            raise ValueError("Failed to load public key from DER bytes") from e

        # Determine maximum allowed message length.
        # The formula for RSA OAEP is: key_size_in_bytes - 2 * (hash digest size) - 2.
        hash_algo = hashes.SHA256()
        max_length = (loaded_public_key.key_size // 8) - 2 * hash_algo.digest_size - 2

        if len(message_bytes) > max_length:
            raise ValueError(
                "Message too long for RSA encryption with a {}-bit key and SHA256 OAEP padding; "
                "maximum is {} bytes, got {} bytes".format(
                    loaded_public_key.key_size, max_length, len(message_bytes)
                )
            )

        encrypted_message = loaded_public_key.encrypt(
            message_bytes,
            asym_padding.OAEP(
                mgf=asym_padding.MGF1(algorithm=hashes.SHA256()),
                algorithm=hashes.SHA256(),
                label=None
            )
        )

        return base64.b64encode(encrypted_message).decode('utf-8')

    @staticmethod
    def decrypt(encrypted: str, private_key: str) -> str:
        if not isinstance(encrypted, str):
            raise TypeError("encrypted must be a string")
        if not isinstance(private_key, str):
            raise TypeError("private_key must be a string")

        # Validate and decode the Base64 encrypted message string
        try:
            encrypted_bytes = base64.b64decode(encrypted)
        except Exception as e:
            raise ValueError("encrypted message is not valid Base64") from e

        # Validate and decode the Base64 private key string
        try:
            der_private_bytes = base64.b64decode(private_key)
        except Exception as e:
            raise ValueError("private_key is not valid Base64") from e

        try:
            loaded_private_key = serialization.load_der_private_key(
                der_private_bytes,
                password=None
            )
        except Exception as e:
            raise ValueError("Failed to load private key from DER bytes") from e

        decrypted_bytes = loaded_private_key.decrypt(
            encrypted_bytes,
            asym_padding.OAEP(
                mgf=asym_padding.MGF1(algorithm=hashes.SHA256()),
                algorithm=hashes.SHA256(),
                label=None
            )
        )

        return decrypted_bytes.decode('utf-8')

    @staticmethod
    def sign(message: str, private_key: str) -> str:
        """
        Signs pre-hashed data using RSA PKCS#1 v1.5 with a SHA-256 prehash.
        
        :param message: A hexadecimal string representation of the SHA-256 hash.
        :param private_key: A Base64-encoded DER private key.
        :return: The Base64-encoded signature.
        """
        if not isinstance(message, str):
            raise TypeError("message must be a string")
        if not isinstance(private_key, str):
            raise TypeError("private_key must be a string")
        
        # Convert the hex string to raw digest bytes.
        try:
            digest = bytes.fromhex(message)
        except Exception as e:
            raise ValueError("Failed to convert message from hex to bytes") from e

        # Decode the Base64 private key.
        try:
            der_private_bytes = base64.b64decode(private_key)
        except Exception as e:
            raise ValueError("private_key is not valid Base64") from e

        # Load the private key.
        try:
            loaded_private_key = serialization.load_der_private_key(
                der_private_bytes,
                password=None
            )
        except Exception as e:
            raise ValueError("Failed to load private key from DER bytes") from e

        # Sign the pre-hashed digest using PKCS#1 v1.5.
        signature = loaded_private_key.sign(
            digest,
            asym_padding.PKCS1v15(),
            Prehashed(hashes.SHA256())
        )
        return base64.b64encode(signature).decode('utf-8')

    @staticmethod
    def verify(signature: str, public_key: str) -> str:
        """
        Recovers the hash from an RSA PKCS#1 v1.5 signature using 
        RSAPublicKey.recover_data_from_signature and returns it as a hex string.
        
        :param signature: The Base64-encoded signature.
        :param public_key: A Base64-encoded DER public key.
        :return: The recovered hash as a hex string.
        """
        if not isinstance(signature, str):
            raise TypeError("signature must be a string")
        if not isinstance(public_key, str):
            raise TypeError("public_key must be a string")
        
        # Decode the Base64 public key.
        try:
            der_public_bytes = base64.b64decode(public_key)
        except Exception as e:
            raise ValueError("public_key is not valid Base64") from e

        # Load the public key.
        try:
            loaded_public_key = serialization.load_der_public_key(der_public_bytes)
        except Exception as e:
            raise ValueError("Failed to load public key from DER bytes") from e

        # Decode the signature from Base64.
        try:
            signature_bytes = base64.b64decode(signature)
        except Exception as e:
            raise ValueError("signature is not valid Base64") from e

        # Recover the padded data using recover_data_from_signature.
        try:
            recovered_data = loaded_public_key.recover_data_from_signature(
                signature_bytes,
                asym_padding.PKCS1v15(),
                algorithm=hashes.SHA256()
            )
        except Exception as e:
            raise ValueError("Failed to recover data from signature") from e

        # Expected DER prefix for SHA-256 DigestInfo:
        sha256_der_prefix = bytes.fromhex("3031300d060960864801650304020105000420")
        if recovered_data.startswith(sha256_der_prefix):
            recovered_digest = recovered_data[len(sha256_der_prefix):]
            if len(recovered_digest) != 32:
                raise ValueError("Recovered hash length is not 32 bytes")
            return recovered_digest.hex()
        elif len(recovered_data) == 32:
            # Likely, the signature was created with Prehashed mode,
            # so the recovered data is just the raw digest.
            return recovered_data.hex()
        else:
            raise ValueError("Recovered data does not have the expected SHA256 DER prefix")

